"use client"

import { Facebook, Instagram, Linkedin, Mail, MapPin, Phone, Youtube } from "lucide-react"
import { Button } from "@/components/ui/button"
import Link from "next/link"
import { useEffect, useState } from "react"
import { createBrowserClient } from "@/lib/supabase/client"

interface Settings {
  phone: string
  email: string
  address: string
  facebook_url: string
  instagram_url: string
  youtube_url: string
  linkedin_url: string
}

export function Footer() {
  const [settings, setSettings] = useState<Settings | null>(null)
  const supabase = createBrowserClient()

useEffect(() => {
  async function fetchSettings() {
    const { data, error } = await supabase
      .from("site_settings")
      .select("key, value")

    if (error) {
      console.error(error)
      return
    }

    const parsed = data.reduce((acc: any, item) => {
      acc[item.key] = item.value
      return acc
    }, {})

    setSettings(parsed)
  }

  fetchSettings()
}, [supabase])



  return (
    <footer id="contato" className="bg-foreground text-background py-16">
      <div className="container mx-auto px-4">
        <div className="grid md:grid-cols-4 gap-12 mb-12">
          {/* Brand */}
          <div>
            <h3 className="text-2xl font-bold mb-4">Uniporá</h3>
            <p className="text-background/80 leading-relaxed mb-6">
              Centro Universitário de Iporá - Construindo o futuro através da educação de qualidade.
            </p>
            <div className="flex gap-3">
  {settings?.facebook_url && (
    <Button
      size="icon"
      className="bg-[#1877F2] text-white border border-[#1877F2] hover:bg-[#166FE5] hover:text-white"
      asChild
    >
      <Link href={settings.facebook_url} target="_blank" rel="noopener noreferrer" aria-label="Facebook">
        <Facebook className="h-5 w-5" />
      </Link>
    </Button>
  )}

  {settings?.instagram_url && (
    <Button
      size="icon"
      className="bg-[#FF722C] text-white border border-[#FF722C] hover:bg-[#E86222] hover:text-white"
      asChild
    >
      <Link href={settings.instagram_url} target="_blank" rel="noopener noreferrer" aria-label="Instagram">
        <Instagram className="h-5 w-5" />
      </Link>
    </Button>
  )}

  {settings?.linkedin_url && (
    <Button
      size="icon"
      className="bg-[#0A66C2] text-white border border-[#0A66C2] hover:bg-[#095BAA] hover:text-white"
      asChild
    >
      <Link href={settings.linkedin_url} target="_blank" rel="noopener noreferrer" aria-label="LinkedIn">
        <Linkedin className="h-5 w-5" />
      </Link>
    </Button>
  )}

  {settings?.youtube_url && (
    <Button
      size="icon"
      className="bg-[#FF0000] text-white border border-[#FF0000] hover:bg-[#E60000] hover:text-white"
      asChild
    >
      <Link href={settings.youtube_url} target="_blank" rel="noopener noreferrer" aria-label="YouTube">
        <Youtube className="h-5 w-5" />
      </Link>
    </Button>
  )}
</div>

          </div>

          {/* Quick Links */}
          <div>
            <h4 className="text-lg font-bold mb-4">Links Rápidos</h4>
            <ul className="space-y-3">
              <li>
                <Link href="#sobre" className="text-background/80 hover:text-background transition-colors">
                  Sobre a Uniporá
                </Link>
              </li>
              <li>
                <Link href="#cursos" className="text-background/80 hover:text-background transition-colors">
                  Cursos
                </Link>
              </li>
              <li>
                <Link href="#ingresso" className="text-background/80 hover:text-background transition-colors">
                  Processo de Ingresso
                </Link>
              </li>
              <li>
                <Link href="#rematricula" className="text-background/80 hover:text-background transition-colors">
                  Rematrícula
                </Link>
              </li>
            </ul>
          </div>

          {/* For Students */}
          <div>
            <h4 className="text-lg font-bold mb-4">Para Alunos</h4>
            <ul className="space-y-3">
              <li>
                <Link href="https://fai.paineldoaluno.com.br/" className="text-background/80 hover:text-background transition-colors">
                  Portal do Aluno
                </Link>
              </li>
              <li>
                <Link href="https://portal.unipora.edu.br/login/index.php" className="text-background/80 hover:text-background transition-colors">
                  Biblioteca Virtual
                </Link>
              </li>
              <li>
                <Link href="#" className="text-background/80 hover:text-background transition-colors">
                  Calendário Acadêmico
                </Link>
              </li>
              <li>
                <Link href="#" className="text-background/80 hover:text-background transition-colors">
                  Segunda Via de Boleto
                </Link>
              </li>
            </ul>
          </div>

          {/* Contact */}
          <div>
            <h4 className="text-lg font-bold mb-4">Contato</h4>
            <ul className="space-y-4">
              <li className="flex gap-3">
                <MapPin className="h-5 w-5 flex-shrink-0 text-background/80" />
                <span className="text-background/80 text-sm">{settings?.address || "Iporá - Goiás, Brasil"}</span>
              </li>
              <li className="flex gap-3">
                <Phone className="h-5 w-5 flex-shrink-0 text-background/80" />
                <span className="text-background/80 text-sm">{settings?.phone || "(64) 3674-0000"}</span>
              </li>
              <li className="flex gap-3">
                <Mail className="h-5 w-5 flex-shrink-0 text-background/80" />
                <span className="text-background/80 text-sm">{settings?.email || "contato@unipora.edu.br"}</span>
              </li>
            </ul>
          </div>
        </div>

        {/* Bottom Bar */}
        <div className="pt-8 border-t border-background/20 text-center text-sm text-background/70">
          <p>© 2026 Uniporá - Centro Universitário de Iporá. Todos os direitos reservados.</p>
        </div>
      </div>
    </footer>
  )
}
